//Source file: H:\middleware\src\Alkindi\BusinessServices\Util\LogWriter.java

package Alkindi.Services.Util;

import javax.jms.*;
import java.rmi.RMISecurityManager;
import javax.naming.NamingException;
import javax.naming.Context;
import javax.naming.InitialContext;
import java.io.FileWriter;
import java.io.IOException;

/* 
$Header: LogWriter.java, 7, 1/2/01 3:05:11 PM, Schwartz, Joe$
$Log: 
 7    Alkindi Development1.6         1/2/01 3:05:11 PM    Schwartz, Joe  
      Removed unnecessary console logging.
 6    Alkindi Development1.5         1/2/01 11:58:25 AM   Schwartz, Joe   Added
      prefix to console ouput.
 5    Alkindi Development1.4         12/28/00 1:07:26 PM  Schwartz, Joe   Added
      Version Control header info.
 4    Alkindi Development1.3         12/27/00 1:43:15 PM  Schwartz, Joe   
 3    Alkindi Development1.2         12/26/00 6:57:14 PM  Schwartz, Joe   
 2    Alkindi Development1.1         12/26/00 1:04:12 PM  Schwartz, Joe   
 1    Alkindi Development1.0         12/26/00 12:14:46 PM Schwartz, Joe   
$
$NoKeywords$
 */

/**
 * Receives log and error messages via JMS and writes them out to files. 
 * The default JNDI name for the topic connection factory is in the field <b>DEFAULT_TOPIC_CNXN_FACTORY</b>.
 * 
 * The default JNDI topic names are in the fields <b>DEFAULT_ERR_TOPIC</b> and <b>DEFAULT_LOG_TOPIC</b>. 
 * The default log file names are in the fields <b>DEFAULT_ERR_FILE</b> and <b>DEFAULT_LOG_FILE</b>. 
 * These can be overridden by specifying various system properties; see the fields' documentation for the property names.
 */
public class LogWriter 
{
	
	/**
	 * The prefix to assign a message that is to be output to the console.
	 */
	private static final String CONSOLE_MSG_PREFIX = "[ALKINDI LOG WRITER]";
	
	/**
	 * Default name of output file for errors. This can be overridden by the sytem property <b>alkindi.errfile</b>.
	 */
	private static final String DEFAULT_ERR_FILE = "default_AK_MW.err";
	
	/**
	 * Defualt JNDI name for the Error Topic in JMS. This can be overridden by the value of the system property <b>alkindi.errtopic</b>.
	 */
	private static final String DEFAULT_ERR_TOPIC = "topic/AlkindiErrTopic";
	
	/**
	 * Default name of output file for logs.This can be overridden by the sytem property <b>alkindi.logfile</b>.
	 */
	private static final String DEFAULT_LOG_FILE = "default_AK_MW.out";
	
	/**
	 * Defualt JNDI name for the Log Topic in JMS. This can be overridden by the value of the system property <b>alkindi.logtopic</b>.
	 */
	private static final String DEFAULT_LOG_TOPIC = "topic/AlkindiLogTopic";
	
	/**
	 * Default JNDI name for JMS Topic Connection Factory. May be overridden with the System property <b>alkindi.TopicConnectionFactory</b>
	 */
	private static final String DEFAULT_TOPIC_CNXN_FACTORY = "TopicConnectionFactory";
	private LogListener errListener = null;
	private TopicSubscriber errSubscriber = null;
	private Topic errTopic = null;
	private TopicConnection jmsConn = null;
	private TopicSession jmsSession = null;
	private LogListener logListener = null;
	private TopicSubscriber logSubscriber = null;
	private Topic logTopic = null;
	
	protected void finalize() throws Throwable 
	{
		if (errSubscriber != null) {
			try { errSubscriber.close(); } catch(Exception e) {}
		}
		if (logSubscriber != null) {
			try {logSubscriber.close(); } catch(Exception e) {}
		}
		if (jmsSession != null) {
			try { jmsSession.close(); } catch(Exception e) {}
		}
		if (jmsConn != null) {
			try { jmsConn.close(); } catch(Exception e) {}
		}
		super.finalize();
	}
	
	/**
	 * Outputs a message to the console. Simply calls System.out.println(), prepending the value of the field CONSOLE_MSG_PREFIX to the message.
	 * @param msg The message to output.
	 * @return void
	 * @roseuid 3A51F60501E4
	 */
	public static final synchronized void consoleOut(final java.lang.String msg) 
	{
		System.out.println(CONSOLE_MSG_PREFIX + " " + msg);
	}
	
	/**
	 * LogWriter Constructor. Constructs instances of LogListener to process log and error messages from the respective topics.
	 * @roseuid 3A48CC7C00EA
	 */
	public LogWriter() throws Exception 
	{
		//	Set error & log file names according to System properties.
		//	If the properties do not exist, set to defaults.
		//
		String errFileName = System.getProperty("alkindi.errfile");
		if (errFileName == null)
			errFileName = DEFAULT_ERR_FILE;
		String logFileName = System.getProperty("alkindi.logfile");
		if (logFileName == null)
			logFileName = DEFAULT_LOG_FILE;
		//	Attempt to open log files for appending.
		//
		try {
			errListener = new LogListener(errFileName);
			logListener = new LogListener(logFileName);
		}
		catch(Exception e) {
			consoleOut("Could not create LogListener objects: " + e.getMessage());
			throw e;
		}

		//	Set JMS Topic & Cnxn Factory JNDI names. Use System properties 
		//	if they exist; defaults otherwise.
		//
		String errTopicName = System.getProperty("alkindi.errtopic");
		if (errTopicName == null)
			errTopicName = DEFAULT_ERR_TOPIC;
		String logTopicName = System.getProperty("alkindi.logtopic");
		if (logTopicName == null)
			logTopicName = DEFAULT_LOG_TOPIC;
		String topicCnxnFactoryName = System.getProperty("alkindi.TopicConnectionFactory");
		if (topicCnxnFactoryName == null)
			topicCnxnFactoryName = DEFAULT_TOPIC_CNXN_FACTORY;
		
		//	Attempt to access JMS topics through JNDI.
		//
		Context ctx = null;
		TopicConnectionFactory qcf = null;
		try {
			ctx = new InitialContext();
			errTopic = (Topic)ctx.lookup(errTopicName);
			logTopic = (Topic)ctx.lookup(logTopicName);
			qcf = (TopicConnectionFactory)ctx.lookup(topicCnxnFactoryName);
			ctx.close();
		}
		catch(Exception ne){
			consoleOut("Could not locate JMS resources in JNDI: " + ne.getMessage());
			ne.printStackTrace();
			if (ctx != null) {
				try { ctx.close(); } catch(Exception e) {}
			}
			throw ne;
		}
		try {

			jmsConn = qcf.createTopicConnection();//"sahra", "spot");
			jmsSession = jmsConn.createTopicSession(false, Session.AUTO_ACKNOWLEDGE);
			errSubscriber = jmsSession.createSubscriber(errTopic);
			logSubscriber = jmsSession.createSubscriber(logTopic);
			jmsConn.start();
			errSubscriber.setMessageListener(errListener);
			logSubscriber.setMessageListener(logListener);
		}
		catch(Exception je) {
			consoleOut("Error initializing JMS resources: " + je.getMessage());
			je.printStackTrace();
			if (jmsConn != null) {
				try { jmsConn.close(); } catch (Exception e) {}
			}
			throw je;
		}
	}
	
	/**
	 * Allows this class to run as a stand-alone program.
	 * @param args The arguments passed to the process.
	 * @return void
	 * @roseuid 3A439A8802BF
	 */
	public static final void main(java.lang.String[] argv) 
	{
		consoleOut("Starting...");
		//	Set new security manager for JNDI access.
		//
		System.setSecurityManager(new RMISecurityManager());
		LogWriter This = null;
		try {
			This = new LogWriter();
		}
		catch(Exception e) {
			consoleOut("Error creating LogWriter: " + e.getMessage());
			e.printStackTrace();
			System.exit(1);
		}
	

		consoleOut("Initialization complete. Waiting for messages.");
		try {
			//	Enter wait state until interruped.
			//
			synchronized(This) {
				This.wait();
			}
		}
		catch (InterruptedException ie) {
			//	Normal exit
			//
			consoleOut("Caught interrupt. Exitiing");
		}
		catch (Exception e) {
			// Error
			//
			consoleOut("Caught exception while in receive loop: " + e.getMessage());
			e.printStackTrace();
		}
	}
	
	/**
	 * Inner class for asynchronous message processing.
	 */
	private class LogListener implements MessageListener 
	{
		
		/**
		 * The IO object used for file output.
		 */
		private FileWriter file = null;
		
		/**
		 * Storage for the requested file name.
		 */
		private String fileName = null;
		
		protected void finalize() throws Throwable 
		{
			if (file != null) {
				try { file.flush(); file.close();	} catch (Exception e) {}
			}
			super.finalize();
		}
		
		/**
		 * Constructor.
		 * @roseuid 3A48B8F003A9
		 */
		public LogListener(final java.lang.String fileToOpen) throws IOException 
		{
			//	Attempt to open the file for appending.
			//
			try {
				file = new FileWriter(fileToOpen, true);
				fileName = fileToOpen;
			}
			catch(IOException ioe) {
				consoleOut("Error opening file " + fileName + ":\n" + ioe.getMessage());
				ioe.printStackTrace();
				throw ioe;
			}
		}
		
		/**
		 * Method from interface MessageListener. Processes a single message. JMS implementation framework will make serial calls to this method.
		 * @param msg The message to process.
		 * @return void
		 * @roseuid 3A48B7350138
		 */
		public void onMessage(Message msg) 
		{
			if (msg instanceof TextMessage) {
				TextMessage tm = (TextMessage)msg;
				try {
		//			consoleOut("GOT MESSAGE: " + tm.getText());
					file.write(tm.getText());
					file.flush();
				}
				catch(IOException ie) {
					consoleOut("Error writing to file " + fileName + ":\n" + ie.getMessage());
				}
				catch(JMSException je) {
					consoleOut("Error getting message text: " + je.getMessage());
				}
			}
			else {
				consoleOut("ERROR: Unexpected message type.");
			}
		}
	}
}
